/*
Project:  CocktailCrafter Hardware Tests (Display + LEDs + Encoder + Pumps)
Author:   Thomas Edlinger for www.edistechlab.com
Date:     21.12.2025
Version:  V1.0
IDE: Arduino IDE 2.3.5 
Required Board (Tools -> Board -> Boards Manager...) 
MCU: ESP32S3 Dev Module / USB Connect on the right side "COM" 
ESP: ESP32-S3-DevKit N16R8 - 
Board: esp32 by Espressif Systems V3.3.5 
Required libraries (sketch -> include library -> manage libraries) 
 - Adafruit GFX Library by Adafruit V1.11.11 
 - Adafruit ST7735 Library by Adafruit V1.10.4 
 - Adafruit NeoPixel by Adafruit V1.12.3 */

#define BATTERY_MODUL 0

#if BATTERY_MODUL
const uint8_t M6_IA_Pin = 42;
const uint8_t VBatt     = 1;
#else
const uint8_t M6_IA_Pin = 1;
const uint8_t VBatt     = 42;
#endif

#include <Adafruit_GFX.h>
#include <Adafruit_ST7735.h>
#include <Adafruit_NeoPixel.h>
#include <SPI.h>

// --- Encoder Pins ---
const uint8_t encoderPin_CLK = 14;
const uint8_t encoderPin_DT = 10;
const uint8_t encoderPin_SW = 41;

// --- Display Pins ---
const uint8_t SPI_TFT_CS = 38;
const uint8_t TFT_DC = 3;
const int     TFT_RST = -1;

// --- LED Setup ---
const uint8_t ledPin = 48;
#define NUM_LEDS 36
Adafruit_NeoPixel leds(NUM_LEDS, ledPin, NEO_GRB + NEO_KHZ800);

// --- Motor Pins ---
const uint8_t M1_IA_Pin = 4;
const uint8_t M1_IB_Pin = 5;
const uint8_t M2_IA_Pin = 6;
const uint8_t M2_IB_Pin = 7;
const uint8_t M3_IA_Pin = 15;
const uint8_t M3_IB_Pin = 16;
const uint8_t M4_IA_Pin = 17;
const uint8_t M4_IB_Pin = 18;
const uint8_t M5_IA_Pin = 8;
const uint8_t M5_IB_Pin = 9;
const uint8_t M6_IB_Pin = 2;
const uint8_t M7_IA_Pin = 40;
const uint8_t M7_IB_Pin = 39;
const uint8_t M8_IA_Pin = 47;
const uint8_t M8_IB_Pin = 21;

// Arrays for Pump Pins
const int motorIA_Pins[] = {M1_IA_Pin, M2_IA_Pin, M3_IA_Pin, M4_IA_Pin, M5_IA_Pin, M6_IA_Pin, M7_IA_Pin, M8_IA_Pin};
const int motorIB_Pins[] = {M1_IB_Pin, M2_IB_Pin, M3_IB_Pin, M4_IB_Pin, M5_IB_Pin, M6_IB_Pin, M7_IB_Pin, M8_IB_Pin};

// Display Object
Adafruit_ST7735 tft = Adafruit_ST7735(SPI_TFT_CS, TFT_DC, TFT_RST);

// --- Encoder Variables ---
int lastEncoderStateCLK;
bool encoderPressed = false;

// ---------- FUNCTION DECLARATIONS ----------
void PumpBegin();
void Pump_on(uint8_t PumpNumber);
void Pump_off(uint8_t PumpNumber);
void TestAllPumps();
void showCenteredText(const char* text, uint8_t textSize = 1, uint16_t color = ST7735_WHITE);
int readEncoderDirection();
bool waitForEncoderPress();
void showFinalScreen();

// ---------- SETUP ----------
void setup() {
  Serial.begin(115200);
  Serial.println("Setup started...");

  // --- Display ---
  tft.initR(INITR_BLACKTAB);
  tft.setRotation(3);
  tft.fillScreen(ST7735_BLACK);

  // --- Encoder ---
  pinMode(encoderPin_CLK, INPUT_PULLUP);
  pinMode(encoderPin_DT, INPUT_PULLUP);
  pinMode(encoderPin_SW, INPUT_PULLUP);
  lastEncoderStateCLK = digitalRead(encoderPin_CLK);

  // --- LEDs ---
  leds.begin();
  leds.clear();
  for (int i = 0; i < NUM_LEDS; i++) leds.setPixelColor(i, leds.Color(255, 0, 0));
  leds.show();

  // --- Pumps ---
  PumpBegin();

  // --- Start Sequence ---
  showCenteredText("CocktailCrafter", 1);
  delay(2000);
  showCenteredText("Turn encoder to the right", 1);

  // Wait for encoder rotation right
  while (readEncoderDirection() != 1) delay(10);
  showCenteredText("Turn encoder left", 1);

  // Wait for encoder rotation left
  while (readEncoderDirection() != -1) delay(10);
  showCenteredText("Push the encoder", 1);

  // Wait for encoder press
  while (!waitForEncoderPress()) delay(10);

  // --- Run Pump Test ---
  TestAllPumps();
  showFinalScreen();
}

void loop() {
  // Wenn der Encoder erneut gedrückt wird -> Pumpentest wiederholen
  if (waitForEncoderPress()) {
    TestAllPumps();
    showFinalScreen();
  }
}

// ---------- FUNCTIONS ----------

void showCenteredText(const char* text, uint8_t textSize, uint16_t color) {
  tft.fillScreen(ST7735_BLACK);
  tft.setTextSize(textSize);
  tft.setTextColor(color, ST7735_BLACK);

  int16_t x = (tft.width() - (strlen(text) * 6 * textSize)) / 2;
  int16_t y = (tft.height() - (8 * textSize)) / 2;
  tft.setCursor(x, y);
  tft.print(text);

  Serial.println(text);
}

int readEncoderDirection() {
  int currentStateCLK = digitalRead(encoderPin_CLK);
  int direction = 0;
  if (currentStateCLK != lastEncoderStateCLK) {
    if (digitalRead(encoderPin_DT) != currentStateCLK) direction = 1;  // right
    else direction = -1;  // left
  }
  lastEncoderStateCLK = currentStateCLK;
  return direction;
}

bool waitForEncoderPress() {
  if (digitalRead(encoderPin_SW) == LOW) {
    delay(50); // debounce
    if (digitalRead(encoderPin_SW) == LOW) {
      while (digitalRead(encoderPin_SW) == LOW) delay(10); // wait for release
      return true;
    }
  }
  return false;
}

void PumpBegin() {
  for (int i = 0; i < 8; i++) {
    pinMode(motorIA_Pins[i], OUTPUT);
    pinMode(motorIB_Pins[i], OUTPUT);
    Pump_off(i);
  }
  Serial.println("All pump pins initialized.");
}

void Pump_on(uint8_t PumpNumber) {
  if (PumpNumber < 8) {
    digitalWrite(motorIA_Pins[PumpNumber], LOW);
    digitalWrite(motorIB_Pins[PumpNumber], HIGH);
    Serial.printf("Pump %d ON\n", PumpNumber + 1);
  }
}

void Pump_off(uint8_t PumpNumber) {
  if (PumpNumber < 8) {
    digitalWrite(motorIA_Pins[PumpNumber], LOW);
    digitalWrite(motorIB_Pins[PumpNumber], LOW);
    Serial.printf("Pump %d OFF\n", PumpNumber + 1);
  }
}

void TestAllPumps() {
  for (int i = 0; i < 8; i++) {
    char msg[10];
    sprintf(msg, "Pump %d", i + 1);
    showCenteredText(msg, 2);
    Pump_on(i);
    delay(3000);
    Pump_off(i);
    delay(500);
  }
}

void showFinalScreen() {
  tft.fillScreen(ST7735_BLACK);
  tft.setTextSize(1);
  tft.setTextColor(ST7735_WHITE, ST7735_BLACK);

  // "CocktailCrafter" oben zentriert
  int16_t x1 = (tft.width() - (strlen("CocktailCrafter") * 6)) / 2;
  tft.setCursor(x1, 20);
  tft.print("CocktailCrafter");

  // Darunter "- Push for repeat -"
  int16_t x2 = (tft.width() - (strlen("- Push for repeat -") * 6)) / 2;
  tft.setCursor(x2, 60);
  tft.print("- Push for repeat -");

  Serial.println("Ready for repeat.");
}
